let popupCreated = false;
let popupVisible = false;
let jQueryLoaded = false;
let pluginInitialized = false;
/**
 * 初始化插件功能
 * @param {string} selector - 内容选择器
 */
async function initializePlugin(selector) {
  // 防止重复初始化
  if (pluginInitialized) {
    console.log('插件已经初始化过，跳过重复初始化');
    return;
  }
  
  console.log('开始初始化插件...');
  
  // 先加载 jQuery
  // 保存选择器到全局变量
  window.currentSelector = selector || '.dad65929';
  
  // 等待DOM准备就绪
  if (document.readyState === 'loading') {
    await new Promise(resolve => {
      document.addEventListener('DOMContentLoaded', resolve);
    });
  }
  
  // 确保jQuery已加载
  if (typeof $ === 'undefined') {
    console.log('等待jQuery加载...');
    await new Promise(resolve => {
      const checkJQuery = setInterval(() => {
        if (typeof $ !== 'undefined') {
          clearInterval(checkJQuery);
          resolve();
        }
      }, 100);
    });
  }
  
  setupMoreActionsButton();
  
  // 标记为已初始化
  pluginInitialized = true;
  console.log('插件初始化完成');
}

/**
 * 设置"更多操作"按钮的添加和监听
 */
function setupMoreActionsButton() {
  // 立即尝试添加按钮
  addMoreActionsButton();
  
  // 添加悬浮球
  addFloatingButton();
  
  // 创建DOM变化观察器
  const observer = new MutationObserver(() => {
    addMoreActionsButton();
  });
  
  // 检查文档是否已经准备好
  if ($('body').length) {
    observer.observe($('body')[0], { childList: true, subtree: true });
  } else {
    // 如果body还不存在，等待它加载
    $(document).ready(() => {
      observer.observe($('body')[0], { childList: true, subtree: true });
    });
  }
}

// 监听插件安装/更新事件
$(document).on('ds_plugin_installed', (event) => {
  initializePlugin(event.detail.selector);
});

// 监听来自背景脚本的初始化消息
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'ping') {
    sendResponse({ pong: true });
    return true;
  }
  
  if (request.action === 'ds_plugin_installed') {
    console.log('收到插件安装初始化消息');
    initializePlugin(request.selector);
    sendResponse({ success: true });
    return true;
  }
  
  // ... 其他现有的消息处理逻辑保持不变
  if (request.action === "togglePopup") {
    // 保存 selector 到全局变量
    window.currentSelector = request.selector;
    
    // 每次接收到新消息时，先将旧的更新信息清除
    window.updateInfo = null;
    
    // 然后保存最新的更新信息(如果有)
    if (request.updateInfo) {
      window.updateInfo = request.updateInfo;
    }
    
    if (typeof popupCreated === 'undefined' || !popupCreated) {
      popupCreated = false;
      popupVisible = false;
      createPopup();
    } else {
      // 更新版本提示状态
      updateVersionNotification();
    }
    togglePopupVisibility();
  } else if (request.action === "closePopup") {
    if (popupVisible) {
      togglePopupVisibility();
    }
  } else if (request.action === "conversionComplete") {
    // 根据 convertType 确定是哪个按钮
    // request 现在应该包含原始的 convertType (e.g. 'doc', 'pdf')
    if (request.originalConvertType === 'word') {
      resetButton($('#export-all-word'));
    } else if (request.originalConvertType === 'pdf') {
      resetButton($('#export-all-pdf'));
    }
    // 旧的: resetButton('convert-button');
  } else if (request.action === "conversionError") {
    if (request.convertType === 'word') {
      resetButton($('#export-all-word'), request.errorMsg);
    } else if (request.convertType === 'pdf') {
      resetButton($('#export-all-pdf'), request.errorMsg);
    }
    // 旧的: resetButton('convert-button', request.errorMsg);
  } else if (request.action === "toggleFloatingButton") {
    // 切换悬浮球显示/隐藏
    const floatingButton = $('#dssxz-floating-button');
    if (floatingButton.length) {
      floatingButton.toggle();
    } else {
      addFloatingButton();
    }
  } else if (request.action === 'updateLoginStatus') {
    // 确保在弹窗内部更新 (这部分逻辑可能需要简化或集中到 updateUserInfo/resetUserInterface)
    if (popupCreated && popupVisible) { // 检查弹窗是否真的创建和可见
    if (request.isLoggedIn) {
        updateUserInfo(request.userData); // 主更新函数
    } else {
        resetUserInterface(); // 主重置函数
      }
    } else { // 如果弹窗未创建或不可见，仍然更新本地存储的状态
      if (request.isLoggedIn) {
          chrome.storage.local.set({ 'wasLoggedIn': true });
        } else {
          chrome.storage.local.set({ 'wasLoggedIn': false });
       }
    }
  }
  // 确保sendResponse在异步操作后被调用，或者返回true
  // 如果某些分支是异步的，要确保它们也处理sendResponse或返回true
  // 对于大部分现有分支，如果它们不调用sendResponse，可以移除它或确保同步返回
  // 对于大部分现有分支，如果它们不调用sendResponse，可以移除它或确保同步返回
  // return true; // 仅在需要异步sendResponse时
});

// 在页面加载完成时初始化
if (document.readyState === 'loading') {
  $(document).ready(() => {
    if (!pluginInitialized) {
      initializePlugin();
    }
  });
} else {
  // 如果页面已加载，立即初始化
  if (!pluginInitialized) {
    initializePlugin();
  }
}

// 添加一个全局标识，防止重复初始化
if (!window.dssxzPluginInitialized) {
  window.dssxzPluginInitialized = true;
  
  // 如果jQuery还没有加载，等待它加载
  if (typeof $ === 'undefined') {
    // 等待jQuery加载
    const checkJQuery = setInterval(() => {
      if (typeof $ !== 'undefined') {
        clearInterval(checkJQuery);
        if (!pluginInitialized) {
          initializePlugin();
        }
      }
    }, 100);
  } else {
    // jQuery已加载，直接初始化
    if (!pluginInitialized) {
      initializePlugin();
    }
  }
}

/**
 * 创建弹窗
 */
function createPopup() {
  // 检查是否已创建弹窗
  if (popupCreated) {
    return;
  }

  // 创建弹窗容器
  const popupDiv = $('<div>', {
    id: 'custom-popup',
    css: { display: 'none' } // 初始不显示
  });
  $('body').append(popupDiv);

  // 加载弹窗内容
  const popupContentUrl = chrome.runtime.getURL('popup_content.html');
  $.get(popupContentUrl)
    .done(html => {
      // 替换图标路径
      const iconUrl = chrome.runtime.getURL('images/icon128.png');
      html = html.replace('images/icon128.png', iconUrl);
      
      popupDiv.html(html);
      
      // 设置事件监听器
      setupEventListeners();
      
      // 立即从后台获取最新的登录状态并更新UI
      chrome.runtime.sendMessage({ action: 'getLatestLoginStatus' }, function(response) {
        if (response && response.isLoggedIn) {
          updateUserInfo(response.userData);
        } else {
          resetUserInterface();
        }
      });
      
      // 标记弹窗已创建
      popupCreated = true;
      console.log("弹窗创建完成");
    })
    .fail(error => {
      console.error('加载弹窗内容失败:', error);
      // 失败时重置标志
      popupCreated = false;
    });
}

// 将事件监听器设置抽取为单独的函数
function setupEventListeners() {
  // 插入图标
  const iconPlaceholder = $('#icon-placeholder');
  if (iconPlaceholder.length) {
    const iconImg = $('<img>', {
      src: chrome.runtime.getURL('images/icon128.png'),
      alt: "DS随心转",
      css: { /* 样式已移至CSS, 这里可以留空或移除 */ }
    });
    iconPlaceholder.append(iconImg);
  }

  // 设置版本更新信息(如果有)
  if (window.updateInfo) {
    const versionUpdateDiv = $('#version-update');
    const latestVersionSpan = $('#latest-version');
    const updateLink = $('#update-link');
    
    if (versionUpdateDiv.length && latestVersionSpan.length && updateLink.length) {
      latestVersionSpan.text(window.updateInfo.latestVersion);
      updateLink.attr('href', window.updateInfo.updateUrl);
      versionUpdateDiv.css('display', 'flex');
    }
  }

  // 设置当前版本号 (从 manifest 获取)
  const currentVersionDisplay = $('#current-version-display');
  if (currentVersionDisplay.length) {
    const manifest = chrome.runtime.getManifest();
    currentVersionDisplay.text(`版本: ${manifest.version}`);
  }

  // 设置前往网站链接
  const websiteLink = $('#website-link');
  if (websiteLink.length) {
    const manifest = chrome.runtime.getManifest();
    const apiBaseUrl = manifest.config.apiBaseUrl;
    websiteLink.attr('href', apiBaseUrl);
  }

  // 登录按钮事件
  $('#login-button').on('click', function() {
    const apiBaseUrl = chrome.runtime.getManifest().config.apiBaseUrl;
    const loginUrl = apiBaseUrl + '/login';
    window.open(loginUrl, '_blank');
    chrome.storage.local.set({ 'loginAttemptTime': Date.now() });
    setTimeout(() => {
      chrome.runtime.sendMessage({ action: "startLoginStatusPolling" });
    }, 1000);
  });

  // 升级VIP按钮事件
  $('#upgrade-vip-button').on('click', function() {
    const apiBaseUrl = chrome.runtime.getManifest().config.apiBaseUrl;
    window.open(apiBaseUrl + '/price', '_blank');
  });

  // 新的导出全部对话按钮事件
  $('#export-all-word').on('click', function() {
    if ($(this).hasClass('disabled')) return;
    disableButton(this, "导出中...");
    convertAllContent('word'); // 'doc' for word, 'all' for entire conversation
  });

  $('#export-all-pdf').on('click', function() {
    if ($(this).hasClass('disabled')) return;
    disableButton(this, "导出中...");
    convertAllContent('pdf'); // 'pdf', 'all' for entire conversation
  });
  
  // 移除旧的导出逻辑 (format selectors and generic convert button)
  // $('.export-button').off('click'); // 移除旧的format selector事件
  // $('#convert-button').off('click'); // 移除旧的通用导出按钮事件

  // 添加关闭按钮事件监听
  $('#close-popup').on('click', function() {
    togglePopupVisibility();
  });

  // 问题反馈按钮事件 (改为链接点击)
  $('#feedback-link').on('click', function(e) {
    e.preventDefault(); // 因为是a标签
  const feedbackInputArea = $('#feedback-input-area');
      feedbackInputArea.css('display', feedbackInputArea.css('display') === 'none' ? 'block' : 'none');
    });

  // 提交反馈按钮事件
  const submitFeedbackButton = $('#submit-feedback-button');
  if (submitFeedbackButton.length) {
    submitFeedbackButton.on('click', function() {
      const feedbackText = $('#feedback-textarea').val();
      const originalText = submitFeedbackButton.text();
      // 禁用提交按钮并更改文本 (可使用disableButton辅助函数)
      submitFeedbackButton.text('提交中...').prop('disabled', true);
      
      chrome.runtime.sendMessage({ 
        action: "submitFeedback", 
        feedbackText: feedbackText
      }, (response) => {
        const feedbackActionDiv = $('.feedback-action');
        // 移除旧的消息
        feedbackActionDiv.find('.feedback-success-message, .feedback-error-message').remove();

        if (response && response.success) {
          const successMsg = $('<span>', {
            class: 'feedback-success-message',
            text: response.message || '感谢您的反馈！'
          });
          feedbackActionDiv.prepend(successMsg);
          $('#feedback-textarea').val('');
          setTimeout(() => {
            successMsg.remove();
            $('#feedback-input-area').css('display', 'none');
          }, 3000);
        } else {
          const errorMsg = $('<span>', {
            class: 'feedback-error-message',
            text: response ? response.message : '提交失败，请稍后重试'
          });
          feedbackActionDiv.prepend(errorMsg);
          setTimeout(() => {
            errorMsg.remove();
          }, 5000);
        }
        submitFeedbackButton.text(originalText).prop('disabled', false);
      });
    });
  }

  // 添加邀请好友按钮事件 (改为链接点击)
  $('#invite-friends-link').on('click', function(e) {
    e.preventDefault();
      const apiBaseUrl = chrome.runtime.getManifest().config.apiBaseUrl;
      window.open(apiBaseUrl + '/user/promotion/index', '_blank');
    });

  // 添加联系客服按钮事件 (改为链接点击)
  $('#contact-link').on('click', function(e) {
    e.preventDefault();
      window.open('https://work.weixin.qq.com/kfid/kfce896a62daeea2ad0', '_blank');
    });

  // "设置导出格式" 链接事件 (如果需要交互)
  $('#export-settings-link').on('click', function(e) {
    e.preventDefault();
    const apiBaseUrl = chrome.runtime.getManifest().config.apiBaseUrl;
    window.open(apiBaseUrl + '/user/profile/set-word-template', '_blank');
  });

  // 保存对话按钮事件
  const saveConversationButton = $('#save-conversation');
  if (saveConversationButton.length) {
    saveConversationButton.on('click', function() {
      if ($(this).hasClass('disabled')) return;
      disableButton(this, "保存中...");
      saveConversation();
    });
  }

  // 确保管理对话链接初始隐藏 (新样式已通过CSS控制)
  // const manageLink = $('#manage-conversations');
  // if (manageLink.length) {
  //   manageLink.css('display', 'none').removeClass('visible');
  // }
}

/**
 * 在页面加载时接收背景脚本的登录状态更新
 */
document.addEventListener('DOMContentLoaded', function() {
  // 向背景脚本发送消息以获取最新的登录状态
  chrome.runtime.sendMessage({ action: 'getLatestLoginStatus' }, function(response) {
    if (response && response.isLoggedIn) {
      updateUserInfo(response.userData);
      // 保存当前登录状态
      chrome.storage.local.set({ 'wasLoggedIn': true });
    } else {
      resetUserInterface();
      // 保存当前登录状态
      chrome.storage.local.set({ 'wasLoggedIn': false });
    }
  });
});

/**
 * 更新用户信息UI
 * @param {Object} userData - 用户数据
 */
function updateUserInfo(userData) {
  $('#user-name').text(userData.username || '已登录用户');
  
  if (userData.isVip) {
    $('#user-hint').text('高级账户已激活，释放AI对话全部潜能');
    $('#vip-badge').show();
    $('#upgrade-vip-button').hide();
  } else {
    $('#user-hint').text('升级高级账户，解锁更多对话管理及导出特权');
    $('#vip-badge').hide();
    $('#upgrade-vip-button').show().text('升级账户'); // 确保文本正确
  }
  
  if (userData.avatar) {
    $("#user-avatar-img").show();
    $('#user-avatar-img').attr('src', userData.avatar);
  }
  
  $('#login-button').hide();
}

/**
 * 重置用户界面到未登录状态
 */
function resetUserInterface() {
  $('#user-name').text('未登录');
  $('#user-hint').text('登录后可使用更多功能');
  $('#user-avatar-img').hide();
  $('#vip-badge').hide();
  
  $('#login-button').show();
  $('#upgrade-vip-button').show().text('升级高级账户'); // 确保按钮文本和显示
  
  // 重置导出次数显示
  $('#export-word-count').text('');
  $('#export-pdf-count').text('');
  $('#export-settings-link .vip-tag-inline').hide();

  // 隐藏登出按钮 (如果之前有的话，新UI中没有明确的登出按钮)
  // $('#logout-button-container').hide();
}

/**
 * 刷新页面内容
 */
function refreshPageContent() {
  // 重新获取对话内容
  const selector = window.currentSelector || '.dad65929';
  const contentElement = $(selector);
  
  if (contentElement.length) {
    // 触发内容刷新
    chrome.runtime.sendMessage({ 
      action: "refreshContent"
    });
  }
}

// 添加禁用按钮的辅助函数
function disableButton(buttonElement, loadingText) {
  const $button = $(buttonElement);
  let textElement;

  // 检查按钮结构以找到正确的文本元素
  if ($button.find('.button-text-main').length) {
    textElement = $button.find('.button-text-main');
  } else if ($button.find('.action-text').length) { // 兼容旧的 #save-conversation
    textElement = $button.find('.action-text');
  } else if ($button.find('.convert-text').length) { // 兼容旧的 #convert-button
     textElement = $button.find('.convert-text');
  } else {
    textElement = $button; // 直接按钮本身
  }

  if (textElement && textElement.length) {
    buttonElement.dataset.originalText = textElement.text();
    textElement.text(loadingText);
  } else {
     buttonElement.dataset.originalText = $button.text();
     $button.text(loadingText);
  }
  
  buttonElement.disabled = true;
  $button.addClass('disabled');
}

// 添加重置按钮状态的辅助函数
function resetButton(buttonIdOrElement, errorMsg) {
  const button = (typeof buttonIdOrElement === 'string') ? $('#' + buttonIdOrElement) : $(buttonIdOrElement);
  
  if (button.length && button.hasClass('disabled')) {
    let textElement;
    // 确定文本元素
    if (button.find('.button-text-main').length) {
      textElement = button.find('.button-text-main');
    } else if (button.find('.action-text').length) {
      textElement = button.find('.action-text');
    } else if (button.find('.convert-text').length) {
      textElement = button.find('.convert-text');
    } else {
      textElement = button;
    }

    const originalText = button[0].dataset.originalText || 
                         (button.is('#save-conversation') ? '云端保存，解锁高级功能' : 
                         (button.is('#export-all-word') ? 'Word' :
                         (button.is('#export-all-pdf') ? 'PDF' : '操作')));

    if (errorMsg) {
      showErrorToast(errorMsg, 5000);
    } 
    if (textElement && textElement.length) textElement.text(originalText);
    else button.text(originalText);
    
    button.prop('disabled', false).removeClass('disabled');
  }
}

// 修改 convertContent 函数
async function convertAllContent(convertType) { // conversationType will be 'all'
  try {
    const selector = window.currentSelector || '.dad65929';
    const contentElement = $(selector);
    const htmlContent = contentElement.html(); // For 'all', backend handles fetching all
    
    // 使用带重试机制的消息发送函数
    await sendMessageWithRetry({ 
      action: "convertAllContent", 
      htmlContent: htmlContent, 
      convertType: convertType, 
    });
    
  } catch (error) {
    console.error('导出内容时出错:', error);
    if (convertType === 'word') resetButton($('#export-all-word'), '导出失败');
    if (convertType === 'pdf') resetButton($('#export-all-pdf'), '导出失败');
  }
}

function togglePopupVisibility() {
  const popupDiv = $('#custom-popup');
  if (!popupDiv.length) return; // 如果弹窗不存在，则直接返回

  popupVisible = !popupVisible; // 切换可见性状态
  popupDiv.css('display', popupVisible ? 'block' : 'none'); // 设置 display 属性
  
  // 当弹窗打开时，重置所有按钮状态和更新提示区域
  if (popupVisible) {
    resetAllButtons();
    // 获取并显示当前版本号
    const currentVersionDisplay = $('#current-version-display');
    if (currentVersionDisplay.length) {
      const manifest = chrome.runtime.getManifest();
      currentVersionDisplay.text(`版本: ${manifest.version}`);
    }
    
    // 使用带重试机制的消息发送
    sendMessageWithRetry({ action: 'getLatestLoginStatus' })
      .then(response => {
        if (response && response.isLoggedIn) {
          updateUserInfo(response.userData);
        } else {
          resetUserInterface();
        }
      })
      .catch(error => {
        console.error('获取登录状态失败:', error);
        resetUserInterface();
      });
  }
}

// 修改重置所有按钮的函数
function resetAllButtons() {
  // 重置导出按钮
  resetButton($('#export-all-word'));
  resetButton($('#export-all-pdf'));
  
  // 重置保存对话按钮
  resetButton($('#save-conversation'));
  
  // 隐藏管理对话链接 (新UI使用.visible类)
  const manageLink = $('#manage-conversations');
  if (manageLink.length) {
    manageLink.removeClass('visible'); // CSS will handle display:none via opacity and height
    // $('.save-tooltip').fadeIn(300); // 旧的tooltip，新UI为 .button-subtext, 通常不隐藏
  }
  // 隐藏反馈区域
  $('#feedback-input-area').css('display', 'none');
  $('#feedback-textarea').val('');
  $('.feedback-action').find('.feedback-success-message, .feedback-error-message').remove();
}

// 添加版本通知更新函数
function updateVersionNotification() {
  const versionUpdateDiv = $('#version-update');
  const latestVersionSpan = $('#latest-version');
  const updateLink = $('#update-link');
  
  if (versionUpdateDiv.length && latestVersionSpan.length && updateLink.length && window.updateInfo) {
    latestVersionSpan.text(window.updateInfo.latestVersion);
    updateLink.attr('href', window.updateInfo.updateUrl);
    versionUpdateDiv.css('display', 'flex');
  } else if (versionUpdateDiv.length) {
    versionUpdateDiv.css('display', 'none');
  }
}

/**
 * 保存对话内容到服务器
 */
async function saveConversation() {
  try {
    const selector = window.currentSelector || '.dad65929';
    const contentElement = $(selector);
    
    if (!contentElement.length) {
      console.error(`未找到选择器 ${selector} 对应的元素`);
      resetButton($('#save-conversation'), '保存失败');
      return;
    }

    // 使用带重试机制的消息发送
    const response = await sendMessageWithRetry({
      action: "saveConversation",
      htmlContent: contentElement.html()
    });
    
    if (response && response.success) {
      resetButton($('#save-conversation'));
      showSuccessToast(response.message || '对话保存成功！');
      showManageConversationsLink();
    } else {
      resetButton($('#save-conversation'), '保存失败');
      showErrorToast(response.message || '保存失败，请重试');
    }
    
  } catch (error) {
    console.error('保存对话时出错:', error);
    resetButton($('#save-conversation'), '保存失败');
    showErrorToast('保存失败: ' + error.message);
  }
}

/**
 * 显示管理对话链接
 */
function showManageConversationsLink() {
  const manageLink = $('#manage-conversations');
  if (manageLink.length) {
    const apiBaseUrl = chrome.runtime.getManifest().config.apiBaseUrl;
    manageLink.attr('href', `${apiBaseUrl}/dialog`);
    manageLink.show();
  }
}

/**
 * 添加右下角悬浮球
 */
function addFloatingButton() {
  // 如果已经存在悬浮球，则不再添加
  if ($('#dssxz-floating-button').length) return;
  
  // 创建悬浮球元素
  const floatingButton = $('<div>', {
    id: 'dssxz-floating-button',
    css: {
      position: 'fixed',
      right: '130px',
      bottom: '130px',
      width: '50px',
      height: '50px',
      borderRadius: '25px',
      boxShadow: '0 4px 8px rgba(0, 0, 0, 0.2)',
      display: 'flex',
      justifyContent: 'center',
      alignItems: 'center',
      cursor: 'pointer',
      zIndex: 9999,
      transition: 'all 0.3s ease'
    }
  });
  console.log('floatingButton', floatingButton);
  
  // 添加图标
  const iconUrl = chrome.runtime.getURL('images/icon128.png');
  const icon = $('<img>', {
    src: iconUrl,
    alt: 'DS随心转',
    css: {
      width: '50px',
      height: '50px',
      borderRadius: '25px'
    }
  });
  
  floatingButton.append(icon);
  
  // 添加鼠标悬停效果
  floatingButton.hover(
    function() {
      $(this).css({
        transform: 'scale(1.1)',
        boxShadow: '0 6px 12px rgba(0, 0, 0, 0.3)'
      });
    },
    function() {
      $(this).css({
        transform: 'scale(1)',
        boxShadow: '0 4px 8px rgba(0, 0, 0, 0.2)'
      });
    }
  );
  
  // 修改点击事件 - 发送消息到背景脚本请求配置并切换弹窗
  floatingButton.on('click', async function() {
    try {
      // 发送消息到背景脚本，请求获取配置并切换弹窗
      const response = await sendMessageWithRetry({
        action: "requestTogglePopup"
      });
      
      if (!response || !response.success) {
        console.error('请求弹窗切换失败:', response?.error);
        // 如果请求失败，仍然尝试显示弹窗（使用默认配置）
        if (typeof popupCreated === 'undefined' || !popupCreated) {
          popupCreated = false;
          popupVisible = false;
          createPopup();
        }
        togglePopupVisibility();
      }
    } catch (error) {
      console.error('悬浮按钮点击处理失败:', error);
      // 发生错误时，仍然尝试显示弹窗（使用默认配置）
      if (typeof popupCreated === 'undefined' || !popupCreated) {
        popupCreated = false;
        popupVisible = false;
        createPopup();
      }
      togglePopupVisibility();
    }
  });
  
  // 将悬浮球添加到页面
  $('body').append(floatingButton);
  
  // 添加拖拽功能
  makeFloatingButtonDraggable(floatingButton);
}

/**
 * 让悬浮球可拖拽
 * @param {jQuery} floatingButton - 悬浮球jQuery对象
 */
function makeFloatingButtonDraggable(floatingButton) {
  let isDragging = false;
  let offsetX, offsetY;
  
  floatingButton.on('mousedown', function(e) {
    isDragging = true;
    
    // 计算偏移量
    offsetX = e.clientX - floatingButton.offset().left;
    offsetY = e.clientY - floatingButton.offset().top;
    
    // 添加移动和松开事件处理
    $(document).on('mousemove', moveHandler);
    $(document).on('mouseup', upHandler);
    
    // 防止默认行为和冒泡
    e.preventDefault();
    e.stopPropagation();
  });
  
  function moveHandler(e) {
    if (!isDragging) return;
    
    // 计算新位置
    let left = e.clientX - offsetX;
    let top = e.clientY - offsetY;
    
    // 确保不超出屏幕范围
    const windowWidth = $(window).width();
    const windowHeight = $(window).height();
    const buttonWidth = floatingButton.outerWidth();
    const buttonHeight = floatingButton.outerHeight();
    
    left = Math.max(0, Math.min(windowWidth - buttonWidth, left));
    top = Math.max(0, Math.min(windowHeight - buttonHeight, top));
    
    // 设置新位置
    floatingButton.css({
      left: left + 'px',
      right: 'auto',
      top: top + 'px',
      bottom: 'auto'
    });
  }
  
  function upHandler() {
    isDragging = false;
    
    // 移除移动和松开事件处理
    $(document).off('mousemove', moveHandler);
    $(document).off('mouseup', upHandler);
    
    // 保存位置到本地存储
    saveFloatingButtonPosition(floatingButton.position());
  }
}

/**
 * 保存悬浮球位置到本地存储
 * @param {Object} position - 包含left和top属性的对象
 */
function saveFloatingButtonPosition(position) {
  chrome.storage.local.set({
    floatingButtonPosition: {
      left: position.left,
      top: position.top
    }
  });
}

/**
 * 从本地存储中恢复悬浮球位置
 * @param {jQuery} floatingButton - 悬浮球jQuery对象
 */
function restoreFloatingButtonPosition(floatingButton) {
  chrome.storage.local.get('floatingButtonPosition', function(data) {
    if (data.floatingButtonPosition) {
      floatingButton.css({
        left: data.floatingButtonPosition.left + 'px',
        top: data.floatingButtonPosition.top + 'px',
        right: 'auto',
        bottom: 'auto'
      });
    }
  });
}

// 优化页面加载后检查登录状态的逻辑

// 在文档可交互时立即检查登录状态
$(document).ready(function() {
  // 通知背景脚本立即检查登录状态
  chrome.runtime.sendMessage({ 
    action: "checkLoginStatus",
    immediate: true  // 表示这是页面加载后的立即检查
  });
});

// 添加页面可见性变化监听，当用户切回标签页时检查状态
document.addEventListener('visibilitychange', function() {
  if (document.visibilityState === 'visible') {
    // 通知背景脚本检查登录状态
    chrome.runtime.sendMessage({ action: "checkLoginStatus" });
  }
});

/**
 * 在指定类名的div元素中添加"更多操作"按钮
 * 页面结构: <div class="ds-flex"><div class="ds-flex _965abe9"></div></div>
 */
function addMoreActionsButton() {

  let responseDivs = document.querySelectorAll(".ds-flex._965abe9");
  responseDivs.forEach((responseDiv) => {
    if (responseDiv.querySelector(".dssxz-export-word")) return;

    const existingCopyButton = responseDiv.querySelector(".ds-icon");

    if (!existingCopyButton) {
      console.log("No existing copy button found");
      return;
    }

    let wordButton = createWordButton();
    wordButton.classList.add("dssxz-export-word");
    responseDiv.append(wordButton);

    wordButton.addEventListener(
      "click",
      async (e) => {
        e.preventDefault();
        e.stopPropagation();
        wordButton.style.color = "#cccccc";
        try {
          existingCopyButton.click();
          await new Promise((resolve) => setTimeout(resolve, 1000));
          const text = await navigator.clipboard.readText();
          if (!text.trim()) {
            throw new Error("复制的内容为空");
          }
          const result = await convertToFormat(text, 'word');
          if (result.success) {
            chrome.runtime.sendMessage({
              action: "downloadFile",
              url: result.data.url
            });
          } else {
            throw new Error(result.message || "转换失败");
          }
        } catch (error) {
          showErrorToast(error.message || "转换失败，请重试");
        }
      },
      true
    );

    let excelButton = createExcelButton();
    excelButton.classList.add("dssxz-export-excel");
    responseDiv.append(excelButton);

    excelButton.addEventListener(
      "click",
      async (e) => {
        e.preventDefault();
        e.stopPropagation();
        excelButton.style.color = "#cccccc";
        try {
          existingCopyButton.click();
          await new Promise((resolve) => setTimeout(resolve, 1000));

          // 从剪贴板获取文本
          const text = await navigator.clipboard.readText();
          
          if (!text.trim()) {
            throw new Error("复制的内容为空");
          }
          
          // 发送文本到API进行转换
          const result = await convertToFormat(text, 'excel');
          
          if (result.success) {
            // 下载文件
            chrome.runtime.sendMessage({
              action: "downloadFile",
              url: result.data.url
            });
          } else {
            throw new Error(result.message || "转换失败");
          }
        } catch (error) {
          showErrorToast(error.message || "转换失败，请重试");
        }
      },
      true
    );

    let imageButton = createImageButton();
    imageButton.classList.add("dssxz-export-image");
    responseDiv.append(imageButton);

    imageButton.addEventListener(
      "click",
      async (e) => {
        e.preventDefault();
        e.stopPropagation();
        
        // 显示加载指示器
        imageButton.style.color = "#cccccc";
        
        try {
          // 点击原始复制按钮将文本复制到剪贴板
          existingCopyButton.click();

          // 等待剪贴板被填充
          await new Promise((resolve) => setTimeout(resolve, 1000));

          // 从剪贴板获取文本
          const text = await navigator.clipboard.readText();
          
          if (!text.trim()) {
            throw new Error("复制的内容为空");
          }
          
          // 发送文本到API进行转换
          const result = await convertToFormat(text, 'image');
          
          if (result.success) {
            // 下载文件
            chrome.runtime.sendMessage({
              action: "downloadFile",
              url: result.data.url
            });
          } else {
            throw new Error(result.message || "转换失败");
          }
        } catch (error) {
          showErrorToast(error.message || "转换失败，请重试");
        }
      },
      true
    );

    let pdfButton = createPdfButton();
    pdfButton.classList.add("dssxz-export-pdf");
    responseDiv.append(pdfButton);

    pdfButton.addEventListener(
      "click",
      async (e) => {
        e.preventDefault();
        e.stopPropagation();
        
        // 显示加载指示器
        pdfButton.style.color = "#cccccc";
        
        try {
          // 点击原始复制按钮将文本复制到剪贴板
          existingCopyButton.click();

          // 等待剪贴板被填充
          await new Promise((resolve) => setTimeout(resolve, 1000));

          // 从剪贴板获取文本
          const text = await navigator.clipboard.readText();
          
          if (!text.trim()) {
            throw new Error("复制的内容为空");
          }
          
          // 发送文本到API进行转换
          const result = await convertToFormat(text, 'pdf');
          
          if (result.success) {
            // 下载文件
            chrome.runtime.sendMessage({
              action: "downloadFile",
              url: result.data.url
            });
          } else {
            throw new Error(result.message || "转换失败");
          }
        } catch (error) {
          showErrorToast(error.message || "转换失败，请重试");
        }
      },
      true
    );
  });

  function createWordButton() {
    const copyButton = $('<div>', {
      class: 'ds-icon-button',
      attr: {
        tabindex: '0'
      },
      css: {
        '--ds-icon-button-size': '20px',
        '--ds-icon-button-hover-color': '#7ea6f6'
      },
      html: getWordButtonIcon()
    })[0];
    return copyButton;
  }
  
  function getWordButtonIcon() {
    
    return `
      <div class="ds-icon" style="font-size: 20px; width: 20px; height: 20px;">
        <svg t="1746778780326" class="icon" viewBox="0 0 1024 1024" version="1.1" xmlns="http://www.w3.org/2000/svg" p-id="2487"><path d="M895.969 336.015l0 575.953c0 26.497-21.498 47.995-47.998 47.995L176.028 959.963c-26.498 0-47.996-21.498-47.996-47.995L128.032 112.033c0-26.498 21.499-47.996 47.996-47.996L623.99 64.037c26.498 0 63.494 15.499 81.994 33.997l155.986 155.987C880.469 272.52 895.969 309.517 895.969 336.015zM831.973 384.011 623.99 384.011c-26.498 0-47.996-21.499-47.996-47.996L575.994 128.032 192.026 128.032l0 767.936 639.946 0L831.972 384.011zM639.99 320.016l187.984 0c-3-8.5-7.5-16.999-11-20.499L660.488 143.03c-3.5-3.5-12-7.999-20.498-10.999L639.99 320.016z" p-id="2488" fill="#4F92FB"></path><path d="M783.309 439.426 612.254 800l-102.52 0-18.691-214.441-1.133 0L390.223 800 286.287 800l-45.596-360.574 103.652 0 11.045 227.695 1.133 0 103.936-227.695 104.219 0 16.992 224.977 1.133 0 82.412-224.977L783.309 439.426z" p-id="2489" fill="#4F92FB"></path></svg>
      </div>
    `;
  }

  function createExcelButton() {
    const copyButton = $('<div>', {
      class: 'ds-icon-button',
      attr: {
        tabindex: '0'
      },
      css: {
        '--ds-icon-button-size': '20px',
        '--ds-icon-button-hover-color': '#7ea6f6'
      },
      html: getExcelButtonIcon()
    })[0];
    return copyButton;
  }
  
  function getExcelButtonIcon() {
    
    return `
      <div class="ds-icon" style="font-size: 20px; width: 20px; height: 20px;">
        <svg t="1746778716210" class="icon" viewBox="0 0 1024 1024" version="1.1" xmlns="http://www.w3.org/2000/svg" p-id="2069" width="200" height="200"><path d="M563.2 1006.933333s-3.413333 0 0 0l-549.546667-102.4c-6.826667-3.413333-13.653333-10.24-13.653333-17.066666V170.666667c0-6.826667 6.826667-13.653333 13.653333-17.066667l546.133334-136.533333c3.413333 0 10.24 0 13.653333 3.413333 3.413333 3.413333 6.826667 6.826667 6.826667 13.653333v955.733334c0 3.413333-3.413333 10.24-6.826667 13.653333-3.413333 3.413333-6.826667 3.413333-10.24 3.413333zM34.133333 873.813333l512 95.573334V54.613333L34.133333 184.32v689.493333z" fill="#4F92FB" p-id="2070"></path><path d="M1006.933333 938.666667h-443.733333c-10.24 0-17.066667-6.826667-17.066667-17.066667s6.826667-17.066667 17.066667-17.066667H989.866667v-785.066666H580.266667c-10.24 0-17.066667-6.826667-17.066667-17.066667s6.826667-17.066667 17.066667-17.066667h426.666666c10.24 0 17.066667 6.826667 17.066667 17.066667v819.2c0 10.24-6.826667 17.066667-17.066667 17.066667zM392.533333 699.733333c-6.826667 0-10.24-3.413333-13.653333-6.826666l-204.8-341.333334c-6.826667-10.24-3.413333-20.48 3.413333-23.893333s17.066667-3.413333 23.893334 6.826667l204.8 341.333333c3.413333 6.826667 3.413333 17.066667-6.826667 23.893333h-6.826667z" fill="#4F92FB" p-id="2071"></path><path d="M187.733333 699.733333c-3.413333 0-6.826667 0-10.24-3.413333-6.826667-3.413333-10.24-13.653333-3.413333-23.893333l204.8-341.333334c3.413333-6.826667 13.653333-10.24 23.893333-6.826666 6.826667 3.413333 10.24 13.653333 6.826667 23.893333l-204.8 341.333333c-6.826667 6.826667-10.24 10.24-17.066667 10.24zM733.866667 938.666667c-10.24 0-17.066667-6.826667-17.066667-17.066667V119.466667c0-10.24 6.826667-17.066667 17.066667-17.066667s17.066667 6.826667 17.066666 17.066667V921.6c0 10.24-6.826667 17.066667-17.066666 17.066667z" fill="#4F92FB" p-id="2072"></path><path d="M989.866667 802.133333H563.2c-10.24 0-17.066667-6.826667-17.066667-17.066666s6.826667-17.066667 17.066667-17.066667H989.866667c10.24 0 17.066667 6.826667 17.066666 17.066667s-6.826667 17.066667-17.066666 17.066666zM1006.933333 665.6h-443.733333c-10.24 0-17.066667-6.826667-17.066667-17.066667s6.826667-17.066667 17.066667-17.066666h443.733333c10.24 0 17.066667 6.826667 17.066667 17.066666s-6.826667 17.066667-17.066667 17.066667zM1006.933333 529.066667H580.266667c-10.24 0-17.066667-6.826667-17.066667-17.066667s6.826667-17.066667 17.066667-17.066667h426.666666c10.24 0 17.066667 6.826667 17.066667 17.066667s-6.826667 17.066667-17.066667 17.066667zM1000.106667 392.533333H573.44c-10.24 0-17.066667-6.826667-17.066667-17.066666s6.826667-17.066667 17.066667-17.066667h426.666667c10.24 0 17.066667 6.826667 17.066666 17.066667s-10.24 17.066667-17.066666 17.066666zM1006.933333 256h-443.733333c-10.24 0-17.066667-6.826667-17.066667-17.066667s6.826667-17.066667 17.066667-17.066666h443.733333c10.24 0 17.066667 6.826667 17.066667 17.066666s-6.826667 17.066667-17.066667 17.066667z" fill="#4F92FB" p-id="2073"></path></svg>
      </div>
    `;
  }

  function createImageButton() {
    const copyButton = $('<div>', {
      class: 'ds-icon-button',
      attr: {
        tabindex: '0'
      },
      css: {
        '--ds-icon-button-size': '20px',
        '--ds-icon-button-hover-color': '#7ea6f6'
      },
      html: getImageButtonIcon()
    })[0];
    return copyButton;
  }
  
  function getImageButtonIcon() {
    
    return `
      <div class="ds-icon" style="font-size: 20px; width: 20px; height: 20px;">
        <svg t="1746778752760" class="icon" viewBox="0 0 1024 1024" version="1.1" xmlns="http://www.w3.org/2000/svg" p-id="2279"><path d="M819.2 96H204.8c-59.733333 0-108.8 49.066667-108.8 108.8v616.533333c0 59.733333 49.066667 108.8 108.8 108.8h616.533333c59.733333 0 108.8-49.066667 108.8-108.8V204.8c-2.133333-59.733333-51.2-108.8-110.933333-108.8zM160 819.2V204.8c0-23.466667 19.2-44.8 44.8-44.8h616.533333c23.466667 0 44.8 19.2 44.8 44.8v388.266667l-125.866666-125.866667c-27.733333-27.733333-76.8-27.733333-104.533334 0l-390.4 384c-4.266667 4.266667-6.4 8.533333-6.4 12.8H204.8c-25.6 0-44.8-19.2-44.8-44.8z m659.2 44.8H324.266667l354.133333-354.133333c2.133333-2.133333 6.4-2.133333 8.533333-2.133334s4.266667 0 8.533334 2.133334l160 160c4.266667 4.266667 8.533333 6.4 12.8 6.4v142.933333c-4.266667 25.6-23.466667 44.8-49.066667 44.8z" fill="#4F92FB" p-id="2280"></path><path d="M375.466667 482.133333c59.733333 0 106.666667-46.933333 106.666666-106.666666s-46.933333-106.666667-106.666666-106.666667-106.666667 46.933333-106.666667 106.666667 49.066667 106.666667 106.666667 106.666666z m0-149.333333c23.466667 0 42.666667 19.2 42.666666 42.666667s-19.2 42.666667-42.666666 42.666666-42.666667-19.2-42.666667-42.666666 19.2-42.666667 42.666667-42.666667z" fill="#4F92FB" p-id="2281"></path></svg>
      </div>
    `;
  }

  function createPdfButton() {
    const copyButton = $('<div>', {
      class: 'ds-icon-button',
      attr: {
        tabindex: '0',
        title: '导出PDF'
      },
      css: {
        '--ds-icon-button-size': '20px',
        '--ds-icon-button-hover-color': '#7ea6f6'
      },
      html: getPdfButtonIcon()
    })[0];
    return copyButton;
  }
  
  function getPdfButtonIcon() {
    
    return `
      <div class="ds-icon" style="font-size: 20px; width: 20px; height: 20px;">
        <svg t="1749116911178" class="icon" viewBox="0 0 1024 1024" version="1.1" xmlns="http://www.w3.org/2000/svg" p-id="1894"><path d="M905.185809 178.844158C898.576738 172.685485 891.19337 165.824412 883.21687 158.436127 860.422682 137.322863 837.434925 116.207791 815.697647 96.487895 813.243072 94.261877 813.243072 94.261877 810.786411 92.037081 781.783552 65.781062 757.590948 44.376502 739.713617 29.293612 729.254178 20.469111 721.020606 13.860686 714.970549 9.501727 710.955023 6.608611 707.690543 4.524745 704.47155 2.998714 700.417679 1.07689 696.638044-0.094029 691.307277 0.005928 677.045677 0.273349 665.6 11.769337 665.6 26.182727L665.6 77.352844 665.6 128.522961 665.6 230.863194 665.6 256.448252 691.2 256.448252 896 256.448252 870.4 230.863194 870.4 998.414942 896 972.829884 230.381436 972.829884C187.90385 972.829884 153.6 938.623723 153.6 896.20663L153.6 26.182727 128 51.767786 588.8 51.767786C602.93849 51.767786 614.4 40.312965 614.4 26.182727 614.4 12.05249 602.93849 0.597669 588.8 0.597669L128 0.597669 102.4 0.597669 102.4 26.182727 102.4 896.20663C102.4 966.91021 159.652833 1024 230.381436 1024L896 1024 921.6 1024 921.6 998.414942 921.6 230.863194 921.6 205.278135 896 205.278135 691.2 205.278135 716.8 230.863194 716.8 128.522961 716.8 77.352844 716.8 26.182727C716.8 39.813762 705.748075 50.91427 692.267725 51.167041 687.705707 51.252584 685.069822 50.435995 682.52845 49.231204 682.259458 49.103682 683.344977 49.796618 685.029451 51.010252 689.779394 54.432502 697.145822 60.34494 706.686383 68.394196 724.009052 83.009121 747.816448 104.072869 776.413589 129.961594 778.850014 132.168064 778.850014 132.168064 781.285216 134.376514 802.876774 153.964212 825.739479 174.96442 848.413564 195.966437 856.350957 203.3185 863.697005 210.144893 870.269888 216.269843 874.209847 219.941299 877.019309 222.565641 878.499674 223.951409 888.81866 233.610931 905.019017 233.081212 914.684179 222.768247 924.349344 212.455283 923.819315 196.264383 913.500326 186.604861 911.981323 185.182945 909.155025 182.542876 905.185809 178.844158ZM102.4 461.128719 0 461.128719 0 896.074709 512 896.074709 1024 896.074709 1024 461.128719 153.6 461.128719 153.6 460.531049 102.4 460.531049 102.4 461.128719ZM208.2 711 208.2 819.2 157.6 819.2 157.6 528 269 528C301.533495 528 327.366571 536.466581 346.5 553.4 365.633429 570.333419 375.2 592.733195 375.2 620.6 375.2 649.133476 365.833427 671.333254 347.1 687.2 328.366573 703.066746 302.133502 711 268.4 711L208.2 711ZM208.2 670.4 269 670.4C287.00009 670.4 300.733286 666.166709 310.2 657.7 319.666714 649.233291 324.4 637.000079 324.4 621 324.4 605.266588 319.600047 592.700047 310 583.3 300.399951 573.899953 287.200083 569.066669 270.4 568.8L208.2 568.8 208.2 670.4ZM419.4 819.2 419.4 528 505.4 528C531.133461 528 553.966566 533.733276 573.9 545.2 593.833434 556.666724 609.266611 572.933229 620.2 594 631.133389 615.066771 636.6 639.199863 636.6 666.4L636.6 681C636.6 708.600139 631.100055 732.866562 620.1 753.8 609.099945 774.733438 593.433436 790.866609 573.1 802.2 552.766564 813.533391 529.466799 819.2 503.2 819.2L419.4 819.2ZM470 568.8 470 778.8 503 778.8C529.533466 778.8 549.89993 770.500083 564.1 753.9 578.30007 737.299917 585.533331 713.466822 585.8 682.4L585.8 666.2C585.8 634.599842 578.933402 610.46675 565.2 593.8 551.466598 577.13325 531.533463 568.8 505.4 568.8L470 568.8ZM854.8 695.8 737.6 695.8 737.6 819.2 687 819.2 687 528 872 528 872 568.8 737.6 568.8 737.6 655.4 854.8 655.4 854.8 695.8Z" fill="#3b82f6" p-id="1895"></path></svg>    
      </div>
    `;
  }
}

/**
 * 将内容转换为指定格式
 * @param {string} text - 要转换的文本内容
 * @param {string} format - 转换格式：'word', 'excel', 'image', 或 'pdf'
 * @returns {Promise<Object>} - 包含转换结果的Promise
 */
async function convertToFormat(text, format) {
  return new Promise((resolve, reject) => {
    // 发送消息给背景脚本处理API请求
    chrome.runtime.sendMessage({
      action: "convertSingleResponse",
      format: format,
      content: text
    }, response => {
      if (chrome.runtime.lastError) {
        reject(new Error(chrome.runtime.lastError.message));
      } else if (response.error) {
        reject(new Error(response.error));
      } else {
        resolve(response);
      }
    });
  });
}

/**
 * 显示Toast提示信息
 * @param {string} message - 要显示的消息
 * @param {number} duration - 显示持续时间（毫秒），默认3000ms
 * @param {string} type - 消息类型：'success', 'error', 'warning', 'info'，默认'info'
 */
function showToast(message, duration = 3000, type = 'info') {
  // 移除已存在的toast
  $('#dssxz-toast').remove();
  
  // 创建toast容器
  const toast = $('<div>', {
    id: 'dssxz-toast',
    class: `dssxz-toast dssxz-toast-${type}`,
    text: message,
    css: {
      position: 'fixed',
      top: '20px',
      right: '20px',
      padding: '12px 20px',
      borderRadius: '6px',
      color: '#fff',
      fontSize: '14px',
      fontWeight: '500',
      zIndex: 10000,
      maxWidth: '300px',
      wordWrap: 'break-word',
      boxShadow: '0 4px 12px rgba(0, 0, 0, 0.15)',
      transform: 'translateX(100%)',
      opacity: '0',
      transition: 'all 0.3s ease-in-out'
    }
  });
  
  // 根据类型设置不同的背景色
  let backgroundColor;
  switch (type) {
    case 'success':
      backgroundColor = '#4CAF50';
      break;
    case 'error':
      backgroundColor = '#f44336';
      break;
    case 'warning':
      backgroundColor = '#ff9800';
      break;
    case 'info':
    default:
      backgroundColor = '#2196F3';
      break;
  }
  
  toast.css('backgroundColor', backgroundColor);
  
  // 添加到页面
  $('body').append(toast);
  
  // 显示动画
  setTimeout(() => {
    toast.css({
      transform: 'translateX(0)',
      opacity: '1'
    });
  }, 10);
  
  // 添加点击关闭功能
  toast.on('click', function() {
    hideToast(toast);
  });
  
  // 自动隐藏
  setTimeout(() => {
    hideToast(toast);
  }, duration);
}

/**
 * 隐藏Toast提示
 * @param {jQuery} toast - Toast元素
 */
function hideToast(toast) {
  if (toast && toast.length) {
    toast.css({
      transform: 'translateX(100%)',
      opacity: '0'
    });
    
    // 动画完成后移除元素
    setTimeout(() => {
      toast.remove();
    }, 300);
  }
}

/**
 * 显示成功提示
 * @param {string} message - 消息内容
 * @param {number} duration - 持续时间
 */
function showSuccessToast(message, duration = 3000) {
  showToast(message, duration, 'success');
}

/**
 * 显示错误提示
 * @param {string} message - 消息内容
 * @param {number} duration - 持续时间
 */
function showErrorToast(message, duration = 4000) {
  showToast(message, duration, 'error');
}

/**
 * 显示警告提示
 * @param {string} message - 消息内容
 * @param {number} duration - 持续时间
 */
function showWarningToast(message, duration = 3500) {
  showToast(message, duration, 'warning');
}

/**
 * 带重试机制的消息发送函数
 * @param {Object} message - 要发送的消息
 * @param {number} maxRetries - 最大重试次数
 * @param {number} retryDelay - 重试延迟时间（毫秒）
 */
async function sendMessageWithRetry(message, maxRetries = 3, retryDelay = 1000) {
  for (let attempt = 1; attempt <= maxRetries; attempt++) {
    try {
      // 检查扩展上下文是否有效
      if (!chrome.runtime || !chrome.runtime.sendMessage) {
        throw new Error('Extension context not available');
      }
      
      // 尝试发送消息
      return await new Promise((resolve, reject) => {
        chrome.runtime.sendMessage(message, (response) => {
          if (chrome.runtime.lastError) {
            reject(new Error(chrome.runtime.lastError.message));
          } else {
            resolve(response);
          }
        });
      });
      
    } catch (error) {
      console.warn(`消息发送失败 (尝试 ${attempt}/${maxRetries}):`, error.message);
      
      if (attempt === maxRetries) {
        // 最后一次尝试失败，抛出错误
        throw new Error(`消息发送失败，已重试 ${maxRetries} 次: ${error.message}`);
      }
      
      // 如果是扩展上下文失效，尝试重新加载页面
      if (error.message.includes('Extension context invalidated') || 
          error.message.includes('Extension context not available')) {
        
        // 等待更长时间让扩展重新初始化
        await new Promise(resolve => setTimeout(resolve, retryDelay * 2));
      } else {
        // 其他错误，正常等待后重试
        await new Promise(resolve => setTimeout(resolve, retryDelay));
      }
    }
  }
}