// 添加获取配置的函数
async function getConfig() {
  try {
    const response = await fetch(chrome.runtime.getManifest().config.apiBaseUrl + '/v11/get-config');
    const data = await response.json();
    if (data.success) {
      console.log("Config data:", data);
      // 返回配置和版本信息
      return {
        selector: data.data.selector,
        version: data.data.version, // 获取服务器返回的最新版本号
        updateUrl: data.data.updateUrl 
      };
    } else {
      return {
        selector: '.dad65929',
        version: null,
        updateUrl: null
      };
    }
  } catch (error) {
    return {
      selector: '.dad65929',
      version: null,
      updateUrl: null
    };
  }
}

// 获取当前版本号
function getCurrentVersion() {
  return chrome.runtime.getManifest().version;
}

// 比较版本号
function isNewerVersion(latestVersion, currentVersion) {
  if (!latestVersion) return false;
  
  const latest = latestVersion.split('.').map(Number);
  const current = currentVersion.split('.').map(Number);
  
  for (let i = 0; i < latest.length; i++) {
    if (latest[i] > (current[i] || 0)) return true;
    if (latest[i] < (current[i] || 0)) return false;
  }
  
  return false; // 相等的情况
}

// 修改 getOrCreateClientId 函数以修复 '.then' 错误
async function getOrCreateClientId() {
  // 使用 Promise 包装 chrome.storage.local.get 确保兼容性
  const data = await new Promise(resolve => {
    chrome.storage.local.get('clientId', resolve);
  });
  let clientId = data.clientId;
  if (!clientId) {
    clientId = crypto.randomUUID(); // 生成 UUID
    // 同样使用 Promise 包装 chrome.storage.local.set
    await new Promise(resolve => {
      chrome.storage.local.set({ clientId: clientId }, resolve);
    });
  }
  return clientId;
}

// 全局变量，确保它们在 initializeApp 之前声明
let userCheckInterval;
let tokenRefreshTimer;

// 初始化函数
async function initializeApp() {
  console.log("Initializing background script at startup...");
  await loadTokensFromStorage(); // 1. 加载存储的令牌
  
  // 2. 立即检查一次登录状态。这可能会更新 authTokens。
  await checkUserLoginStatus(); 
  
  // 3. 设置令牌刷新定时器。它依赖于 authTokens 的当前状态。
  setupTokenRefreshTimer();
  
  // 4. 设置定期检查用户登录状态
  if (userCheckInterval) { // 清除任何可能存在的旧定时器
    clearInterval(userCheckInterval);
  }
  userCheckInterval = setInterval(checkUserLoginStatus, 30000); // 每30秒检查一次
  console.log('用户登录状态的定期检查已启动 (每30秒)。');
}

// 在后台脚本启动时调用初始化函数
initializeApp();

// 将配置请求和弹窗切换逻辑独立出来
async function handleTogglePopupRequest(tab) {
  try {
    // 如果是目标网站，获取配置信息
    const config = await getConfig();
    const currentVersion = getCurrentVersion();
    const hasUpdate = isNewerVersion(config.version, currentVersion);
    
    // 发送消息给内容脚本切换弹窗
    chrome.tabs.sendMessage(tab.id, { 
      action: "togglePopup", 
      selector: config.selector,
      updateInfo: hasUpdate ? {
        currentVersion: currentVersion,
        latestVersion: config.version,
        updateUrl: config.updateUrl
      } : null
    });
  } catch (error) {
    console.error('处理弹窗切换请求时出错:', error);
  }
}

async function checkRelevance(url) {
  if(url == ''|| url == null || url == undefined ) return false;
  const relevantDomains = ['chat.deepseek.com'];
  const urlObj = new URL(url);
  return relevantDomains.some(domain => urlObj.hostname.includes(domain));
}

chrome.action.onClicked.addListener(async (tab) => {
  // 检查当前页面是否是 deepseek 官网
  const isRelevant = await checkRelevance(tab.url);
  if (!isRelevant) {
    chrome.scripting.executeScript({
      target: { tabId: tab.id },
      func: () => {
        const confirmed = confirm('【DS随心转】需要在DeepSeek官网使用，是否立即跳转？');
        if (confirmed) {
          window.open('https://chat.deepseek.com/', '_blank');
        }
      }
    });
  }
  await handleTogglePopupRequest(tab);
});

// 监听来自 content script 的消息
chrome.runtime.onMessage.addListener(function(request, sender, sendResponse) {
  if (request.action === "convertAllContent") {
    const htmlContent = request.htmlContent;
    const convertType = request.convertType;
    const tabId = sender.tab.id;

    switch (convertType) {
      case "word":
        convertAllToFile(htmlContent, 'word', tabId);
        sendResponse();
        break;
      case "pdf":
        convertAllToFile(htmlContent, 'pdf', tabId);
        sendResponse();
        break;
      default:
        console.error("Background: Unknown export type:", convertType);
    }
  }

  if (request.action === "downloadFile") {
    chrome.downloads.download({
      url: request.url
    }, function(downloadId) {
      if (chrome.runtime.lastError) {
        console.error("下载文件时出错:", chrome.runtime.lastError.message);
        sendResponse({ success: false, error: chrome.runtime.lastError.message });
      } else {
        console.log("文件下载已启动，下载ID:", downloadId);
        sendResponse({ success: true, downloadId: downloadId });
      }
    });
    return true; // 保持sendResponse有效
  }

  if (request.action === "convertSingleResponse") {
    convertSingleResponse(request.format, request.content)
      .then(response => sendResponse(response))
      .catch(error => sendResponse({ success: false, error: error.message }));
    return true; // 保持sendResponse有效
  }

  // 处理获取最新登录状态的请求
  if (request.action === 'getLatestLoginStatus') {
    // 从存储中获取当前的登录状态和用户数据
    chrome.storage.local.get(['userLoggedIn', 'userData'], function(result) {
      if (result.userLoggedIn && result.userData) {
        sendResponse({
          isLoggedIn: true,
          userData: result.userData,
          hasValidToken: isAccessTokenValid()
        });
      } else {
        sendResponse({
          isLoggedIn: false,
          userData: null,
          hasValidToken: false
        });
      }
    });
    return true; // 保持sendResponse有效
  }

  // 处理保存对话请求
  if (request.action === "saveConversation") {
    saveConversationDialog(request.htmlContent, sender.tab.id)
      .then(result => sendResponse(result))
      .catch(error => sendResponse({ 
        success: false, 
        message: error.message || '保存失败，请重试'
      }));
    return true; // 保持sendResponse有效
  }

  // 处理提交反馈请求
  if (request.action === "submitFeedback") {
    submitUserFeedback(request.feedbackText)
      .then(result => sendResponse(result))
      .catch(error => sendResponse({ 
        success: false, 
        message: error.message || '提交反馈失败，请稍后重试'
      }));
    return true; // 保持sendResponse有效
  }

  // 处理来自内容脚本的弹窗切换请求
  if (request.action === "requestTogglePopup") {
    handleTogglePopupRequest(sender.tab)
      .then(() => sendResponse({ success: true }))
      .catch(error => {
        console.error('处理弹窗切换请求失败:', error);
        sendResponse({ success: false, error: error.message });
      });
    return true; // 保持sendResponse有效
  }
});

async function convertAllToFile(htmlContent, convertType, tabId) {
  const apiBaseUrl = chrome.runtime.getManifest().config.apiBaseUrl;
  const apiUrl = `${apiBaseUrl}/v11/convert-all`;
  const clientId = await getOrCreateClientId();
  
  try {
    // 使用授权请求函数
    const response = await authorizedFetch(apiUrl, {
      method: "POST",
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({ 
        htmlContent: htmlContent, 
        convertType: convertType, 
        clientId: clientId
      }),
    });
    
    if (!response.ok) {
      throw new Error(`响应错误: ${response.success}`);
    }
    
    const data = await response.json();
    
    if (data.success) {
      chrome.downloads.download({
        url: data.data.url
      }, function() {
        chrome.tabs.sendMessage(tabId, {
          action: "closePopup"
        });
      });
    } else {
      console.error("Background: Error during convertToFile:", data.message || '转换失败');
      chrome.tabs.sendMessage(tabId, {
        action: "conversionError",
        convertType: convertType,
        errorMsg: data.message || '转换失败'
      });
    }
  } catch (error) {
    console.error("Background: Network error during convertToFile:", error);
    chrome.tabs.sendMessage(tabId, {
      action: "conversionError",
      convertType: convertType,
      errorMsg: error.message || '网络请求失败'
    });
  }
}

//安装成功后自动跳转到指定页面
chrome.runtime.onInstalled.addListener(async function(details) {
    if (details.reason === "install") {
        // 扩展首次安装时，直接在所有相关标签页中注入代码
        chrome.tabs.query({}, async (tabs) => {
            for (const tab of tabs) {
                if (tab.id && await checkRelevance(tab.url)) {
                    // 先注入 jQuery
                    await chrome.scripting.executeScript({
                        target: { tabId: tab.id },
                        files: ['lib/jquery-3.7.1.min.js']  // 注入 jQuery 库
                    }).catch(err => console.error('注入 jQuery 失败:', err));

                    // 再注入内容脚本
                    await chrome.scripting.executeScript({
                        target: { tabId: tab.id },
                        files: ['js/content.js']  // 注入内容脚本
                    }).catch(err => console.error('注入内容脚本失败:', err));
                }
            }
        });
        // 同时向所有已打开的DeepSeek页面注入内容脚本并初始化
        await initializeExistingTabs();
        // 扩展首次安装时打开指定页面
        chrome.tabs.create({
          url: chrome.runtime.getManifest().config.apiBaseUrl+'/guidev11'
      });
    } else if (details.reason === "update") {
        // 扩展更新时也初始化已打开的页面
        await initializeExistingTabs();
    }
});

/**
 * 初始化所有已打开的DeepSeek标签页
 */
async function initializeExistingTabs() {
    try {
        // 查询所有DeepSeek标签页
        const tabs = await chrome.tabs.query({
            url: "*://chat.deepseek.com/*"
        });
        
        for (const tab of tabs) {
            try {
                // 检查标签页是否完全加载
                if (tab.status === 'complete') {
                    await initializeSingleTab(tab.id);
                }
            } catch (error) {
            }
        }
    } catch (error) {
    }
}

/**
 * 初始化单个标签页
 * @param {number} tabId - 标签页ID
 */
async function initializeSingleTab(tabId) {
    try {
        // 检查标签页是否还存在
        const tab = await chrome.tabs.get(tabId);
        if (!tab) {
            console.log(`标签页 ${tabId} 不存在，跳过初始化`);
            return;
        }
        
        // 先检查内容脚本是否已经注入
        try {
            const response = await chrome.tabs.sendMessage(tabId, { action: 'ping' });
            if (response && response.pong) {
                const config = await getConfig();
                chrome.tabs.sendMessage(tabId, {
                    action: 'ds_plugin_installed',
                    selector: config.selector
                });
                return;
            }
        } catch (e) {
            // 内容脚本未注入，继续注入流程
        }
        
        // 注入jQuery库
        await chrome.scripting.executeScript({
            target: { tabId: tabId },
            files: ['lib/jquery-3.7.1.min.js']
        });
        
        // 注入CSS样式
        await chrome.scripting.insertCSS({
            target: { tabId: tabId },
            files: ['css/content.css']
        });
        
        // 注入内容脚本
        await chrome.scripting.executeScript({
            target: { tabId: tabId },
            files: ['js/content.js']
        });
        
        // 等待脚本加载完成
        await new Promise(resolve => setTimeout(resolve, 1000));
        
        // 获取配置并发送初始化消息
        const config = await getConfig();
        
        // 发送初始化消息，带重试机制
        let retryCount = 0;
        const maxRetries = 3;
        
        while (retryCount < maxRetries) {
            try {
                await chrome.tabs.sendMessage(tabId, {
                    action: 'ds_plugin_installed',
                    selector: config.selector
                });
                break;
            } catch (error) {
                retryCount++;
                if (retryCount < maxRetries) {
                    await new Promise(resolve => setTimeout(resolve, 500));
                } else {
                    console.error(`标签页 ${tabId} 初始化消息发送失败，已达到最大重试次数:`, error);
                }
            }
        }
        
    } catch (error) {
        console.error(`初始化标签页 ${tabId} 时发生错误:`, error);
    }
}


// 添加令牌管理功能

// 存储令牌信息
let authTokens = {
  access_token: null,
  access_token_expire_time: 0,
  refresh_token: null,
  refresh_token_expire_time: 0
};

// 添加一个变量来存储登录状态轮询定时器
let loginStatusPollingInterval;

// 初始化扩展时加载令牌
// chrome.runtime.onInstalled.addListener(async () => { // 这部分逻辑已被 initializeApp 和修改后的 onInstalled 取代
// ...
// });

/**
 * 从存储中加载令牌
 */
async function loadTokensFromStorage() {
  try {
    const data = await chrome.storage.local.get(['authTokens']);
    if (data.authTokens) {
      authTokens = data.authTokens;
    }
  } catch (error) {
    console.error('加载令牌失败:', error);
  }
}

/**
 * 保存令牌到存储
 */
async function saveTokensToStorage() {
  try {
    await chrome.storage.local.set({ authTokens });
  } catch (error) {
  }
}

/**
 * 检查令牌是否有效
 * @returns {boolean} 令牌是否有效
 */
function isAccessTokenValid() {
  if (!authTokens.access_token) return false;
  
  // 检查令牌是否过期（提前5分钟认为已过期，确保操作安全）
  const currentTime = Math.floor(Date.now() / 1000);
  return authTokens.access_token_expire_time > (currentTime + 300);
}

/**
 * 检查刷新令牌是否有效
 * @returns {boolean} 刷新令牌是否有效
 */
function isRefreshTokenValid() {
  if (!authTokens.refresh_token) return false;
  
  const currentTime = Math.floor(Date.now() / 1000);
  return authTokens.refresh_token_expire_time > (currentTime + 300);
}

/**
 * 刷新访问令牌
 * @returns {Promise<boolean>} 刷新是否成功
 */
async function refreshAccessToken() {
  if (!isRefreshTokenValid()) {
    return false;
  }
  try {
    const apiBaseUrl = chrome.runtime.getManifest().config.apiBaseUrl;
    const clientId = await getOrCreateClientId();
    const response = await fetch(`${apiBaseUrl}/refresh-token`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json'
      },
      credentials: 'include',
      body: JSON.stringify({
        accessToken: authTokens.access_token,
        refreshToken: authTokens.refresh_token,
        clientId: clientId
      })
    });
    
    if (!response.ok) {
      throw new Error(`刷新令牌请求失败: ${response.status}`);
    }
    
    const data = await response.json();
    
    if (data.success && data.data && data.data.tokens) {
      // 更新令牌
      authTokens = {
        ...authTokens,
        ...data.data.tokens
      };
      
      // 保存到存储
      await saveTokensToStorage();
      return true;
    } else {
      throw new Error(data.message || '刷新令牌失败');
    }
  } catch (error) {
    return false;
  }
}

/**
 * 获取有效的访问令牌
 * @returns {Promise<string|null>} 有效的访问令牌或null
 */
async function getValidAccessToken() {
  // 如果访问令牌有效，直接返回
  if (isAccessTokenValid()) {
    return authTokens.access_token;
  }
  
  // 尝试刷新访问令牌
  const refreshSuccess = await refreshAccessToken();
  
  if (refreshSuccess) {
    return authTokens.access_token;
  }
  
  // 令牌刷新失败，返回null
  return null;
}

/**
 * 使用授权令牌进行API请求
 * @param {string} url - 请求URL
 * @param {Object} options - 请求选项
 * @returns {Promise<Object>} 响应对象
 */
async function authorizedFetch(url, options = {}) {
  // 获取有效的访问令牌
  const token = await getValidAccessToken();
  
  // 如果没有有效的令牌，尝试检查登录状态
  if (!token) {
    await checkUserLoginStatus();
  }
  
  // 添加授权头
  const headers = {
    ...options.headers || {},
    'Authorization': `Bearer ${token}`
  };
  
  try {
    // 发送请求
    const response = await fetch(url, {
      ...options,
      headers,
      credentials: 'include'
    });
    
    // 处理401错误（未授权）
    if (response.status === 401) {
      // 令牌可能已过期，尝试刷新
      if (await refreshAccessToken()) {
        // 刷新成功，使用新令牌重试请求
        return authorizedFetch(url, options);
      } else {
        // 刷新失败，可能需要重新登录
        // 清除令牌和用户信息
        await clearAuthData();
        // 通知所有标签页用户已登出
        broadcastLoginStatus(false, null);
        throw new Error('请先登录后再操作');
      }
    }
    
    return response;
  } catch (error) {
    console.error('授权请求失败:', error);
    throw error;
  }
}

/**
 * 清除所有授权数据
 */
async function clearAuthData() {
  authTokens = {
    access_token: null,
    access_token_expire_time: 0,
    refresh_token: null,
    refresh_token_expire_time: 0
  };
  
  await chrome.storage.local.remove(['authTokens', 'userLoggedIn', 'userData']);
}

/**
 * 检查用户登录状态 - 在背景脚本中集中处理
 */
async function checkUserLoginStatus() {
  
  const apiBaseUrl = chrome.runtime.getManifest().config.apiBaseUrl;
  const clientId = await getOrCreateClientId();
  
  try {
    const response = await fetch(`${apiBaseUrl}/check-status`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      credentials: 'include',
      body: JSON.stringify({ clientId: clientId })
    });
    
    if (!response.ok) {
      throw new Error('网络请求失败: ' + response.status);
    }
    
    const data = await response.json();
    
    if (data.success && data.data && data.data.isLoggedIn) {
      // 保存令牌
      if (data.data.tokens) {
        authTokens = data.data.tokens;
        await saveTokensToStorage();
      }
      
      // 提取用户信息
      const userData = data.data.user || {};
      
      // 用户已登录，存储登录状态到本地存储
      await chrome.storage.local.set({ 
        userLoggedIn: true, 
        userData: userData
      });
      
      // 通知所有活动标签页更新UI
      broadcastLoginStatus(true, userData);
      return true;
    } else {
      // 用户未登录，清除所有授权数据
      await clearAuthData();
      
      // 通知所有活动标签页更新UI
      broadcastLoginStatus(false, null);
      return false;
    }
  } catch (error) {
    console.error('背景脚本: 检查登录状态时出错:', error);
    
    // 在网络问题时，尝试从本地读取缓存的登录状态
    try {
      const result = await chrome.storage.local.get(['userLoggedIn', 'userData']);
      if (result.userLoggedIn && result.userData) {
        // 通知标签页使用缓存的登录状态
        broadcastLoginStatus(true, result.userData, true);
        return true;
      }
    } catch (err) {
      console.error('读取缓存用户信息失败:', err);
    }
    return false;
  }
}

/**
 * 向所有活动标签页广播登录状态
 * @param {boolean} isLoggedIn - 是否已登录
 * @param {Object} userData - 用户数据
 * @param {boolean} fromCache - 是否来自缓存
 */
function broadcastLoginStatus(isLoggedIn, userData, fromCache = false) {
  chrome.tabs.query({}, (tabs) => {
    tabs.forEach(tab => {
      // 只向匹配的域名发送消息
      if (tab.url && tab.url.includes('deepseek.com')) {
        chrome.tabs.sendMessage(tab.id, {
          action: 'updateLoginStatus',
          isLoggedIn: isLoggedIn,
          userData: userData,
          hasValidToken: isAccessTokenValid(),
          fromCache: fromCache
        }).catch(err => {
          // 处理标签页可能未加载完成的情况
        });
      }
    });
  });
}

/**
 * 转换单条回答
 * @param {string} format - 转换格式
 * @param {string} content - 回答内容
 * @returns {Promise<Object>} 转换结果
 */
async function convertSingleResponse(format, content) {
  const apiBaseUrl = chrome.runtime.getManifest().config.apiBaseUrl;
  let endpoint = '';
  
  // 确定正确的端点
  switch(format) {
    case 'word':
      endpoint = '/v11/convert-to-word';
      break;
    case 'excel':
      endpoint = '/v11/convert-to-excel';
      break;
    case 'image':
      endpoint = '/v11/convert-to-image';
      break;
    case 'pdf':
      endpoint = '/v11/convert-to-pdf';
      break;
    default:
      throw new Error('不支持的格式');
  }
  
  try {
    // 获取客户端ID
    const clientId = await getOrCreateClientId();
    
    // 使用授权请求函数 - 不需要手动添加令牌，authorizedFetch会处理
    const response = await authorizedFetch(`${apiBaseUrl}${endpoint}`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json'
      },
      body: JSON.stringify({
        answer: content,
        clientId: clientId
      })
    });
    
    if (!response.ok) {
      throw new Error(`响应错误: ${response.status}`);
    }
    
    // 解析响应
    const result = await response.json();
    
    if (result.success) {
      return result;
    } else {
      throw new Error(result.message || '转换失败');
    }
  } catch (error) {
    console.error(`${format}转换请求失败:`, error);
    throw error;
  }
}

/**
 * 设置令牌自动刷新定时器
 */
function setupTokenRefreshTimer() {
  // 清除现有定时器
  if (tokenRefreshTimer) {
    clearTimeout(tokenRefreshTimer);
  }
  
  // 如果没有令牌或令牌已经过期，不设置定时器
  if (!isAccessTokenValid()) {
    return;
  }
  
  // 计算令牌过期前15分钟的时间
  const currentTime = Math.floor(Date.now() / 1000);
  const expiresIn = authTokens.access_token_expire_time - currentTime;
  const refreshDelay = Math.max(0, (expiresIn - 900) * 1000); // 提前15分钟刷新，转换为毫秒
  
  // 设置新的定时器
  tokenRefreshTimer = setTimeout(async () => {
    const success = await refreshAccessToken();
    if (success) {
      // 刷新成功，设置下一次刷新
      setupTokenRefreshTimer();
    } else {
    }
  }, refreshDelay);
  
}

// 监听开始登录状态轮询的消息
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  // 其他现有的消息处理...
  
  if (request.action === 'startLoginStatusPolling') {
    startLoginStatusPolling();
    sendResponse({success: true});
    return true; // 保持sendResponse有效
  }
  
  // 处理停止轮询的消息
  if (request.action === 'stopLoginStatusPolling') {
    stopLoginStatusPolling();
    sendResponse({success: true});
    return true; // 保持sendResponse有效
  }
});

/**
 * 开始轮询检查登录状态
 * 用户在新标签页登录后使用
 */
function startLoginStatusPolling() {
  // 先停止任何现有的轮询
  stopLoginStatusPolling();
  
  // 立即检查一次登录状态
  checkUserLoginStatus();
  
  // 设置轮询间隔，每2秒检查一次登录状态
  loginStatusPollingInterval = setInterval(async () => {
    // 获取登录尝试时间
    const data = await chrome.storage.local.get('loginAttemptTime');
    const loginAttemptTime = data.loginAttemptTime || 0;
    
    // 检查是否已经过去了5分钟（300000毫秒）
    // 如果超过5分钟，停止轮询
    if (Date.now() - loginAttemptTime > 300000) {
      stopLoginStatusPolling();
      return;
    }
    
    // 检查登录状态
    const isLoggedIn = await checkUserLoginStatus();
    
    // 如果用户已登录，停止轮询
    if (isLoggedIn) {
      stopLoginStatusPolling();
    }
  }, 2000); // 每2秒检查一次
  
}

/**
 * 停止轮询检查登录状态
 */
function stopLoginStatusPolling() {
  if (loginStatusPollingInterval) {
    clearInterval(loginStatusPollingInterval);
    loginStatusPollingInterval = null;
  }
}

/**
 * 保存对话到服务器
 * @param {string} htmlContent - HTML内容
 * @param {number} tabId - 标签页ID
 * @returns {Promise<Object>} 保存结果
 */
async function saveConversationDialog(htmlContent, tabId) {
  const apiBaseUrl = chrome.runtime.getManifest().config.apiBaseUrl;
  const apiUrl = `${apiBaseUrl}/v11/save-dialog`;
  const clientId = await getOrCreateClientId();
  
  try {
    // 使用授权请求函数
    const response = await authorizedFetch(apiUrl, {
      method: "POST",
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({ 
        content: htmlContent,
        platform: 'DeepSeek',
        clientId: clientId
      }),
    });
    
    if (!response.ok) {
      throw new Error(`响应错误: ${response.status}`);
    }
    
    const data = await response.json();
    
    if (data.success) {
      return {
        success: true,
        message: data.message || '对话保存成功！',
        dialogUrl: data.data && data.data.url ? data.data.url : `${apiBaseUrl}/dialog`
      };
    } else {
      console.error("Background: 保存对话失败:", data.message || '保存失败');
      return {
        success: false,
        message: data.message || '保存失败'
      };
    }
  } catch (error) {
    console.error("Background: 保存对话网络请求失败:", error);
    throw error;
  }
}

/**
 * 提交用户反馈到服务器
 * @param {string} feedbackContent - 用户反馈内容
 * @returns {Promise<Object>} 提交结果
 */
async function submitUserFeedback(feedbackContent) {
  const apiBaseUrl = chrome.runtime.getManifest().config.apiBaseUrl;
  const apiUrl = `${apiBaseUrl}/v1/feedback`;
  const clientId = await getOrCreateClientId();
  const currentVersion = getCurrentVersion();
  
  try {
    // 使用授权请求函数
    const response = await authorizedFetch(apiUrl, {
      method: "POST",
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({ 
        content: feedbackContent,
        clientId: clientId,
        version: currentVersion
      }),
    });
    
    if (!response.ok) {
      throw new Error(`响应错误: ${response.status}`);
    }
    
    const data = await response.json();
    
    if (data.success) {
      return {
        success: true,
        message: data.message || '感谢您的反馈！我们会认真处理您的建议。'
      };
    } else {
      console.error("Background: 提交反馈失败:", data.message || '提交失败');
      return {
        success: false,
        message: data.message || '提交失败，请稍后重试'
      };
    }
  } catch (error) {
    console.error("Background: 提交反馈网络请求失败:", error);
    
    // 检查是否是网络连接问题
    if (error.message.includes('Failed to fetch') || error.message.includes('网络')) {
      return {
        success: false,
        message: '网络连接异常，请检查网络后重试'
      };
    }
    
    throw error;
  }
}
